/************************************************************************/
/* 									*/
/* FileFunc.cc:	   Implementation of the FileFunc class			*/
/*		   This class implements the basic functions for a	*/
/*		   file selction dialogue				*/
/*		   Roland Krause 1998					*/
/*									*/
/************************************************************************/

// --- Include ----------------------------------------------------------

#include <stdio.h>
#include <unistd.h>			// Function getcwd()
#include <stdlib.h>			// Function qsort()
#include <malloc.h>			// Traditional memory management
#include <pwd.h>			// Convert user ID into a string
#include <X11/Intrinsic.h>		// Header of Xt
#include <X11/xpm.h>			// We need XPM functions

#include "FileFunc.h"			// Class definition of this class

#include "pixmaps/file.xpm"		// Icon for files
#include "pixmaps/folder.xpm"		// Icon for folders
#include "pixmaps/tool.xpm"		// Icon for executables
#include "pixmaps/shadefile.xpm"	// Icon for not accessible files
#include "pixmaps/shadefold.xpm"	// Icon for not accessible folders

// --- Constants used to create file list entries -----------------------

#define LIST_LEN	36		// Length of entry without file name
#define FNAME_LEN	14		// Minimum length of file name
#define FNAME_IN_LIST	12		// Position of file name entry in list


// --- Implementation of protected class methods ------------------------

// **********************************************************************
//
// GetIcon():	Determine the needed icon for the given file
//
// Arguments:	FileName  - Name of the file
//		FileFlags - Mode flags determined with stat()
//
// **********************************************************************

Pixmap FileFunc::GetIcon(char *FileName, unsigned short FileFlags)
{
  // In the first step we check, whether a file is a directory or not.
  // If the file is accessible and executable we use the directory icon.
  // If the directory is not accessible we use the icon for this case.

  if (FileFlags & S_IFDIR)
    if (access(FileName, X_OK) == 0) return(folderPix);
    else return(shadefoldPix);
  else

    // File is not a directory:
    // If the file is executable, we use the program icon,
    // else the file icon for readable or unreadable files.

    if (access(FileName, R_OK) == 0)
      if (FileFlags & (S_IEXEC|S_IXGRP|S_IXOTH)) return(toolPix);
      else return(filePix);
    else return(shadefilePix);
}


// **********************************************************************
//
// GetClip():	Determine the clip mask for a given icon
//
// Arguments:	Icon - The icon, we need a clip mask for
//
// **********************************************************************

Pixmap FileFunc::GetClip(Pixmap Icon)
{
  if (Icon == filePix) return(fileClip);
  if (Icon == folderPix) return(folderClip);
  if (Icon == toolPix) return(toolClip);
  if (Icon == shadefilePix) return(shadefileClip);
  if (Icon == shadefoldPix) return(shadefoldClip);
  return(XtUnspecifiedPixmap);
}


// **********************************************************************
//
// FreeFileList(): Free the Directory structure
//
// Arguments:	   None
//
// **********************************************************************

void FileFunc::FreeFileList(void)
{
  int i = 0;

  if (Directory != NULL)
  {
    while (Directory[i].string != NULL)	// Release all entries
    {
       free(Directory[i].string);
       i++;
    }
    free(Directory);			// Release the structure
    Directory = NULL;
  }
}


// **********************************************************************
//
// CheckFNames(): Determine the length of the longest file name
//		  in the current directory
//
// Arguments:	  dir - File handle to read the directory
//			The directory has to be rewinded !
//
// **********************************************************************

int FileFunc::CheckFNames(DIR *dir)
{
  struct dirent *dp;		// Entry in directory file
  int MaxLen = FNAME_LEN;	// Length of longest file name
  int  Len;

  while ((dp = readdir(dir)) != NULL)
  {
     Len = strlen(dp->d_name);
     if (Len > MaxLen) MaxLen = Len;
  }
  rewinddir(dir);
  return(MaxLen);
}

// **********************************************************************
//
// BuildListEntry():	Determine the length of the longest file name
//			in the current directory
//
// Arguments:		String   - The result
//			Name     - The file name
//			FNameLen - Number of characters for the file name
//				   in 'String'
//			FileInfo - Access rights (from stat())
//
// **********************************************************************

void FileFunc::BuildListEntry(char *String, char *Name,
			      int FNameLen, struct stat *FileInfo)
{
   struct passwd *usr;			// Conversion of user ID into ASCII
   char		 FileName[PATH_LEN],	// Buffer for the file name
		 UserName[PATH_LEN],	// and for the name of the owner
		 Blanks[PATH_LEN];	// Used to fill the file name string

   // Generate a string to fill up the file name to the required length

   (void)memset(Blanks, ' ', FNameLen);
   Blanks[FNameLen] = '\0';

   // Build a file name string which has the minimum length of FNameLen
   // characters. Determine the owner of the file

   strcpy(FileName, Name);
   strcat(FileName, Blanks);
   if ((usr = getpwuid(FileInfo->st_uid)) != NULL) strcpy(UserName, usr->pw_name);
   else *UserName = '\0';
   strcat(UserName, Blanks);

   // Put everything together to a file list entry

   sprintf(String, " %c%c%c%c%c%c%c%c%c  %-.*s %-.10s %lu",
	FileInfo->st_mode & S_IREAD  ? 'r' : '-',
	FileInfo->st_mode & S_IWRITE ? 'w' : '-',
	FileInfo->st_mode & S_IEXEC  ? 'x' : '-',
	FileInfo->st_mode & S_IRGRP  ? 'r' : '-',
	FileInfo->st_mode & S_IWGRP  ? 'w' : '-',
	FileInfo->st_mode & S_IXGRP  ? 'x' : '-',
	FileInfo->st_mode & S_IROTH  ? 'r' : '-',
	FileInfo->st_mode & S_IWOTH  ? 'w' : '-',
	FileInfo->st_mode & S_IXOTH  ? 'x' : '-',
	FNameLen, FileName, UserName, FileInfo->st_size);
}


// **********************************************************************
//
// Compare():	Compare two entries of the type XawIconList
//		in alphabetical order.
//
// Arguments:	ent1     - First entry to compare
//		ent2     - Second entry to compare
//		LenFName - Length of longest possible file name
//
// Result:	-1 : ent1 < ent2
//		 0 : ent1 = ent2
//		 1 : ent1 > ent2
//
// **********************************************************************

int FileFunc::Compare(XawIconList ent1, XawIconList ent2, int LenFName)
{
  if ((ent1.bitmap == folderPix) || (ent1.bitmap == shadefoldPix))
    if ((ent2.bitmap != folderPix) && (ent2.bitmap != shadefoldPix)) return (-1);
    else return (strncmp(&(ent1.string[FNAME_IN_LIST]), &(ent2.string[FNAME_IN_LIST]), LenFName));
  else
    if ((ent2.bitmap == folderPix) || (ent2.bitmap == shadefoldPix)) return (1);
    else return (strncmp(&(ent1.string[FNAME_IN_LIST]), &(ent2.string[FNAME_IN_LIST]), LenFName));
}


// **********************************************************************
//
// SortFList():	Sort the files entries of the XawIconList structure
//		in an order defined by Compare().
//		This is the bidirectional bubble sort algorithm.
//
// Arguments:	list        - the file list
//		NoOfEntries - Number of used entries in the list
//		LenFName    - Length of longest file name in the list
//
// **********************************************************************

void FileFunc::SortFList(XawIconList *list, int NoOfEntries, int LenFName)
{
  XawIconList T;
  int	      j, MaxValue, MinValue;
  Boolean     flipped;

  MinValue = -1;
  MaxValue = NoOfEntries-1;

  while (MinValue < MaxValue)
  {
     flipped = FALSE;
     MinValue++;
     MaxValue--;

     for (j = MinValue; j < MaxValue; j++)
     {
	if (Compare(list[j], list[j+1], LenFName) > 0)
	{
	  T = list[j];
	  list[j] = list[j+1];
	  list[j+1] = T;
	  flipped = TRUE;
	}
     }
     if (flipped == FALSE) return;
     flipped = FALSE;

     for (j = MaxValue; j > MinValue; j--)
     {
	if (Compare(list[j], list[j+1], LenFName) > 0)
	{
	  T = list[j];
	  list[j] = list[j+1];
	  list[j+1] = T;
	  flipped = TRUE;
	}
     }
     if (flipped == FALSE) return;
  }
}


// --- Implementation of public class methods ---------------------------

// **********************************************************************
//
// FileFunc():	 The constructor of this class
//		 Do some initializations
//
// Arguments:	 none
//
// **********************************************************************

FileFunc::FileFunc(void)
{
   // We have to initialize the current working directory

   if (getcwd(CurrentPath, PATH_LEN) == NULL) CurrentPath[0] = '\0';
   Directory = NULL;
}


// **********************************************************************
//
// BuildFileList(): Create a file list (type XawIconList *) for a
//		    icon list widget. In case of errors this method
//		    returns a null pointer.
//
// Arguments:	 none
//
// **********************************************************************

XawIconList *FileFunc::BuildFileList(void)
{
   DIR           *dir;			// Pointer for the directory
   struct dirent *dp;			// Entry directory file
   struct stat   fileinfo;		// File informations
   char		 FileName[PATH_LEN];	// Buffer for file names
   int		 Files;			// Number of files in directory
   int		 LenFName;		// Length of the longest file name

   FreeFileList();
   if ((Directory = (XawIconList *)malloc(sizeof(XawIconList))) != NULL)
   {
      Directory[0].bitmap   = XtUnspecifiedPixmap;
      Directory[0].clipMask = XtUnspecifiedPixmap;
      Directory[0].string = NULL;
      Files = 0;
      if ((dir = opendir(CurrentPath)) != NULL)
      {
	 // Determine the length of the longest file name

	 LenFName = CheckFNames(dir);

	 // Read all files of the current directory

	 while ((dp = readdir(dir)) != NULL)
	 {
	   sprintf(FileName, "%s/%s", CurrentPath, dp->d_name);
	   if (stat(FileName, &fileinfo) == 0)
	   {
	      // Check if regular file or directory. All other
	      // files (devices and so on) are ignored

	      if (S_ISDIR(fileinfo.st_mode) || S_ISREG(fileinfo.st_mode))
	      {
		// File found: Allocate memory and build a new entry
		// for the file list

		Files++;
		Directory = (XawIconList *)realloc((char *)Directory, (Files+1) * sizeof(XawIconList));
		Directory[Files].string   = NULL;
		Directory[Files].bitmap   = XtUnspecifiedPixmap;
		Directory[Files].clipMask = XtUnspecifiedPixmap;
		if ((Directory[Files-1].string = (char *)malloc(LIST_LEN+LenFName)) != NULL)
		{
		  // Create a new entry for the file list with the file name
		  // an icon for this file and the clip mask for this icon.

		  BuildListEntry(Directory[Files-1].string, dp->d_name, LenFName, &fileinfo);
		  Directory[Files-1].bitmap   = GetIcon(FileName, fileinfo.st_mode);
		  Directory[Files-1].clipMask = GetClip(Directory[Files-1].bitmap);
		}
		else	// No memory to create the list entry
		{
		  FreeFileList();
		  return (NULL);
		}
	      }
	   }
	 }
	 // Ok: Close the directory and sort the list entries

	 closedir(dir);
	 SortFList(Directory, Files, LenFName);
	 return (Directory);
      }
      // Error: Unable to open the directory

      free(Directory);
      Directory = NULL;
    }
    return (NULL);
}


// **********************************************************************
//
// GetFileName(): Build a complete filename from CurrentPath
//		  and the file index
//
// Arguments:	  Index - Index of directory entry
//
// **********************************************************************

char *FileFunc::GetFileName(int Index)
{
  static char	Filename[PATH_LEN];
  char		*p, *q;

  strcpy(Filename, CurrentPath);	// Copy the path name
  p = Filename;	
  while (*p != '\0') p++;		// Search for end of string
  if (Filename[1] != '\0') *p++ = '/';	// root directory ?
  q = &(Directory[Index].string[FNAME_IN_LIST]);
  while (*q != ' ') *p++ = *q++;	// Copy the file name behind
  *p = '\0';				// Mark the end of the string
  return(Filename);
}


// **********************************************************************
//
// CreateIcons(): The second 'constructor' of this class
//		  Creates all pixmaps
//
// Arguments:	  shell - The root shell widget of this application
//		  This widget must be realized to call this method !
//
// **********************************************************************

void FileFunc::CreateIcons(Widget shell)
{
   Window   win;
   int	    ScrNum;

// Create the pixmaps for the file list

   display = XtDisplay(shell);
   win	   = XtWindow(shell);

   ScrNum    = XDefaultScreen(display);
   IconDepth = XDefaultDepthOfScreen(XScreenOfDisplay(display,ScrNum));

   (void)XpmCreatePixmapFromData(display,win,folder_xpm,&folderPix,&folderClip,NULL);
   (void)XpmCreatePixmapFromData(display,win,shadefold_xpm,&shadefoldPix,&shadefoldClip,NULL);
   (void)XpmCreatePixmapFromData(display,win,file_xpm,&filePix,&fileClip,NULL);
   (void)XpmCreatePixmapFromData(display,win,shadefile_xpm,&shadefilePix,&shadefileClip,NULL);
   (void)XpmCreatePixmapFromData(display,win,tool_xpm,&toolPix,&toolClip,NULL);
}


// **********************************************************************
//
// CheckFileType(): Determine the file type of an directory entry
//
// Arguments:	    Index - Index of directory entry
//
// **********************************************************************

XeFileType FileFunc::CheckFileType(int Index)
{
  // Normal file

  if ((Directory[Index].bitmap == filePix) ||
      (Directory[Index].bitmap == shadefilePix) ||
      (Directory[Index].bitmap == toolPix)) return(FF_FILE);

  // Directory

  if (Directory[Index].bitmap == folderPix) return(FF_DIRECTORY);

  // Unknown entry

  return(FF_FAULT);
}


// **********************************************************************
//
// ChangeDir():	Changes into the directory, found with Index.
//		If the directory name is '.' nothing happens.
//		If the name is '..', we go one step upwards in the
//		directory hierarchy.
//		
// Arguments:	Index - Index of directory entry
//
// **********************************************************************

void FileFunc::ChangeDir(int Index)
{
   char *p, *q, PathName[PATH_LEN];

   // Copy the selected path name from the directory entry

   p = PathName;
   q = &(Directory[Index].string[FNAME_IN_LIST]);
   while (*q != ' ') *p++ = *q++;
   *p = '\0';

   if (strcmp(PathName, ".") != 0)
   {
      if (strcmp(PathName, "..") == 0)
      {
	 // If we are not in the root directory, we go one
	 // step upwards. We have to delete the last part
	 // of the path name.

	 p = strrchr(CurrentPath, '/');
	 if ((p != NULL) && (p != CurrentPath)) *p = '\0';
	 else CurrentPath[1] = '\0';
      }
      else	// Add Pathname to CurrentPath
      {
	 if (CurrentPath[1] != '\0') strcat(CurrentPath, "/");
	 strcat(CurrentPath, PathName);
      }
   }
}


// **********************************************************************
//
// ResetDir():	 Go one step upwards in the directory hierarchy
//		 This method is used, if we try changed into a directory
//		 that does not exist anymore.
//
// Arguments:	 none
//
// **********************************************************************

void FileFunc::ResetDir(void)
{
  char *p;

  // If we are not in the root directory, we go one
  // step upwards. We have to delete the last part
  // of the path name.

  p = strrchr(CurrentPath, '/');
  if ((p != NULL) && (p != CurrentPath)) *p = '\0';
  else CurrentPath[1] = '\0';
}


// **********************************************************************
//
// ~FileFunc():	 The destructor of this class
//		 Destroy all pixmaps
//
// Arguments:	 none
//
// **********************************************************************

FileFunc::~FileFunc(void)
{
   XFreePixmap(display, filePix);		// Release all pixmaps
   XFreePixmap(display, shadefilePix);
   XFreePixmap(display, folderPix);
   XFreePixmap(display, shadefoldPix);
   XFreePixmap(display, toolPix);
   XFreePixmap(display, fileClip);		// Release all clip masks
   XFreePixmap(display, shadefileClip);
   XFreePixmap(display, folderClip);
   XFreePixmap(display, shadefoldClip);
   XFreePixmap(display, toolClip);

// Release the directory list if it exists

   FreeFileList();
}


