/************************************************************************/
/* 									*/
/* HandleTar.cc:   This class handles the communication with tar	*/
/*									*/
/*		   Roland Krause 1998					*/
/*									*/
/************************************************************************/

// --- Include ----------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>			// unlink(), rmdir()
#include <sys/stat.h>			// struct stat for file infos
#include <string.h>			// String functions
#include <malloc.h>			// Traditional memory management

#include "HandleTar.h"			// Class definition of this class


// --- Implementation of private class methods --------------------------

// **********************************************************************
//
// CheckContentList():	Try to read the first entries of the content
//			list using a tar command
//			This method is meant for internal usage from
//			CheckContent() !
//
// Arguments:		tarCommand - The tar command string
//
// **********************************************************************

int HandleTar::CheckContentList(char *tarCommand)
{
  int   len, OldListLength;
  char  TextLine[TAR_PATH_LEN];

  if ((TarStream = popen(tarCommand, "r")) != NULL)
  {
     // Now we have to read the tar output line by line
     // We only read a few entries !

     while ((ContLineCounter < TAR_CHECK_FILES) &&
	    (fgets(TextLine, TAR_PATH_LEN-1, TarStream) != NULL))
     {
	if (*TextLine != 'd')	// Ignore directory entries
	{
	   ContLineCounter++;

	   // Copy the new entry and expand the content string

	   len = strlen(TextLine);
	   OldListLength = ListLength;
	   ListLength += len;
	   if (ContentList == NULL) ContentList = (char *)malloc(len + 1);
	   else ContentList = (char *)realloc(ContentList, ListLength + 1);

	   if (ContentList != NULL)
	     (void)strcpy(&ContentList[OldListLength], TextLine);
	   else
	   {
	     DeleteContent();	// Error: No memory
	     return(0);
	   }
         }
      }
  }
  return(ContLineCounter);
}

// --- Implementation of public class methods ---------------------------

// **********************************************************************
//
// HandleTar():  The constructor of this class
//		 Get the name of the tar file and do some initializations
//
// Arguments:	 FileName - Name of the tar file
//
// **********************************************************************

HandleTar::HandleTar(char *FName)
{
   char *p, *q;

   TarStream	   = NULL;
   ContentList	   = NULL;
   Compressed	   = Unknown;
   ContLineCounter = 0;
   ListLength	   = 0;
   ReadLines	   = 0;

   // To work with the tar archive, we need for some tar commands
   // the absolute path name. That is the reason we build an absolute
   // name here if an relative is given. Note that expressions like ~/name
   // or ../../name are resolved by the shell.

   if (*FName == '/') (void)strcpy(FileName, FName);
   else
   {
     (void)getcwd(FileName, TAR_PATH_LEN);
     (void)strcat(FileName, "/");
     (void)strcat(FileName, FName);
   }
}

// **********************************************************************
//
// ~HandleTar():  The destructor of this class
//
// Arguments:	  None
//
// **********************************************************************

HandleTar::~HandleTar(void)
{
   DeleteContent();
}

// **********************************************************************
//
// DeleteContent():	Delete resources allocated by the content list
//
// Arguments:		None
//
// **********************************************************************

void HandleTar::DeleteContent(void)
{
   if (TarStream != NULL) (void)pclose(TarStream);
   if (ContentList != NULL) free(ContentList);
   TarStream	   = NULL;
   ContentList	   = NULL;
   ContLineCounter = 0;
   ListLength	   = 0;
   ReadLines	   = 0;
}

// **********************************************************************
//
// CheckContent(): Determine the content of the current tar archive.
//		   Returns 0 if the file is no tar archive.
//		   This method sets 'Compressed' to Yes or No, if
//		   a tar archive is detected.
//
//		   This method allocates memory, which has to
//		   be freed by DeleteContent() !
//
// Arguments:	   list - Temporary result of the content list
//
// Result:	   0 - Content list is now complete or empty
//		 > 0 - Absolute number of entries read from archive
//
// **********************************************************************

int HandleTar::CheckContent(char **list)
{
   char CmdString[TAR_PATH_LEN];

   // We have to start tar to get the content list of the tar
   // archive: At first we create the tar command string.

   DeleteContent();
   ReadLines = TAR_READ_FILES;	// The number of lines for the first ReadContent()

   (void)sprintf(CmdString, TAR_LIST_COMP, FileName);

   if (CheckContentList(CmdString) == 0)
   {
      // File ist not a compressed tar file: Try to read it
      // as an ordinary tar file

      (void)sprintf(CmdString, TAR_LIST, FileName);
      if (CheckContentList(CmdString) > 0) Compressed = NotCompressed;
   }
   else Compressed = IsCompressed;

   // Set the return values

   *list = ContentList;

   if (ContLineCounter < TAR_CHECK_FILES) ContLineCounter = 0;
   return(ContLineCounter);
}

// **********************************************************************
//
// ReadContent(): Try to read the next entries of the content list
//
// Arguments:	  list - Temporary result of the content list
//
// Result:	  0 - Content list is now complete
//		> 0 - Absolute number of entries read from archive
//
// **********************************************************************

int HandleTar::ReadContent(char **list)
{
  int   len, OldListLength;
  char  TextLine[TAR_PATH_LEN];

  if (ContLineCounter > 0)
  {
     // Now we have to read the tar output line by line
     // We only read a few entries !

     while (fgets(TextLine, TAR_PATH_LEN-1, TarStream) != NULL)
     {
	if (*TextLine != 'd')	// Ignore directory entries
	{
	   ContLineCounter++;

	   // Copy the new entry and expand the content string

	   len = strlen(TextLine);
	   OldListLength = ListLength;
	   ListLength += len;
	   ContentList = (char *)realloc(ContentList, ListLength + 1);
	   (void)strcpy(&ContentList[OldListLength], TextLine);

	   // Check if we have to break our work

	   if (ContLineCounter >= ReadLines)
	   {
	      ReadLines *= 2;		// This value has to reach the next time
	      *list = ContentList;
	      return (ContLineCounter);
	   }
         }
      }
  }
  *list = ContentList;
  return(0);
}

// **********************************************************************
//
// GetFile():	Extract a file from the current tar archive and
//		load it into memory.
//
// Arguments:	File - Name of the file in the archive
//
//		In the case of success, the user has to free the
//		allocated memory via delete
//
// **********************************************************************

char *HandleTar::GetFile(char *File)
{
   char		TarCommand[TAR_PATH_LEN], FilePath[TAR_PATH_LEN];
   FILE		*fd;
   struct stat	FileInfo;
   char 	*Data = NULL, *ptr;

   switch(Compressed)	// Generate a command string for tar
   {
      case IsCompressed:
		(void)sprintf(TarCommand, TAR_EXTR_COMP, FileName, File);
		break;

      case NotCompressed: 
		(void)sprintf(TarCommand, TAR_EXTR, FileName, File);
		break;

      default:	return(NULL);
   }
   // Now we call tar

   (void)system(TarCommand);

   // In the next step we try to open the tmp file
   // and read it into the memory

   (void)sprintf(FilePath, "/tmp/%s", File);  
   if ((fd = fopen(FilePath,"r")) != NULL)
   {
       if ((stat(FilePath, &FileInfo) != -1)    &&
           ((Data = new char[FileInfo.st_size + 1]) != NULL) &&
           (fread(Data, FileInfo.st_size, 1, fd) != 0)) Data[FileInfo.st_size] = '\0';
       else
       {
	  if (Data != NULL) delete(Data);		// An error occurs
	  Data = NULL;
       }
       // Now we have to remove the tmp file with its
       // directory structure if it exists

       (void)unlink(FilePath);				// Remove the file
       for (;;)
       {
	  if ((ptr = strrchr(FilePath, '/')) == NULL) break;
	  *ptr = '\0';					// Delete the last part of the path
	  if (strcmp(FilePath, "/tmp") == 0) break;	// We don't delete /tmp itself !
	  (void)rmdir(FilePath);
       }
   }
   return(Data);
}

// **********************************************************************
//
// ExtractFiles(): Extract files from the current tar archive
//
// Arguments:	   FileList - List of file names
//
// **********************************************************************

void HandleTar::ExtractFiles(char *FileList, char *Path)
{
   char	*TarCommand;
   int   StrLen;

   // Determine the length of the needed string: The string becomes a little
   // to long because of the %s in TAR_EXTR_C_PATH.

   StrLen = strlen(Path) + strlen(FileName) + strlen(FileList) + sizeof(TAR_EXTR_C_PATH);

   if ((TarCommand = (char *)malloc(StrLen)) != NULL)
   {
     switch(Compressed)	// Generate a command string for tar
     {
        case IsCompressed:
		(void)sprintf(TarCommand, TAR_EXTR_C_PATH, Path, FileName, FileList);
		(void)system(TarCommand);
		break;

        case NotCompressed: 
		(void)sprintf(TarCommand, TAR_EXTR_PATH, Path, FileName, FileList);
		(void)system(TarCommand);
		break;
     }
     free(TarCommand);
   }
}

