/************************************************************************/
/* 									*/
/* FileSelect.cc: Implementation of the FileSelect class		*/
/*		  This is the file handling of this application		*/
/*		  Roland Krause 1998, 1999				*/
/*									*/
/************************************************************************/

// --- Include ----------------------------------------------------------

#include <stdio.h>
#include <malloc.h>

// Header of Xt

#include <X11/Intrinsic.h>		// Header of Xt
#include <X11/StringDefs.h>		// Ressource definitions

// Header of XawPlus

#include <X11/Shell.h>			// Shell widgets
#include <X11/XawPlus/Paned.h>		// The panel widget
#include <X11/XawPlus/Form.h>		// The form widget
#include <X11/XawPlus/Viewport.h>	// The viewport for the file list
#include <X11/XawPlus/IconList.h>	// The list widget for the file list
#include <X11/XawPlus/List.h>		// The list widget
#include <X11/XawPlus/AsciiText.h>	// A text widget for the selected file
#include <X11/XawPlus/Label.h>		// The label widget
#include <X11/XawPlus/Command.h>	// The command widget for the buttons

#include "FileSelect.h"			// Class definition of this class


// --- Implementation of work procedures --------------------------------

// **********************************************************************
//
// ReadTarList(): Asyncronous read of the tar output for the
//		  archive content. This procedure will be installed
//		  from the Ok() callback if needed.
//
// Arguments:	  clientData - A pointer to the file selector object
//
// **********************************************************************

Boolean ReadTarList(XtPointer clientData)
{
  FileSelect *fs = (class FileSelect *)clientData;
  char	     *FileList;

  if (fs->tar->ReadContent(&FileList) > 0)
  {
     fs->UpdateTarList(FileList);	// Update the list and make shure
     return(False);			// that we come back soon here
  }
  fs->UpdateTarList(FileList);		// Tar list is complete
  fs->FinishTarList();
  fs->tar->DeleteContent();
  return(True);
}


// --- Implementation of callbacks --------------------------------------

// The callbacks are not part of this class ! They are only a part of
// this module. That is the reason why all callbacks works on an
// object of the class FileSelect. Since the callbacks are defined
// static, they are not visibile outside of this module.

// **********************************************************************
//
// Cancel():	Close this popup without any action
//
// Arguments:	clientData - A pointer to the file selector object
//
// **********************************************************************

void Cancel(Widget w, XtPointer clientData, XtPointer callData)
{
  FileSelect	*fs = (class FileSelect *)clientData;	
  fs->PopDown();
}

// **********************************************************************
//
// Ok():	Open the selected tar archive and determine the list
//		of entries.
//
// Arguments:	clientData - A pointer to the file selector object
//
// **********************************************************************

void Ok(Widget w, XtPointer clientData, XtPointer callData)
{
  FileSelect	*fs = (class FileSelect *)clientData;

  fs->OpenArchive(NULL);
}

// **********************************************************************
//
// SetFileEntry(): Put the selected file name into text widget
//
// Arguments:	clientData - A pointer to the file selector object
//		callData   - A pointer to the selected entry
//
// **********************************************************************

void SetFileEntry(Widget w, XtPointer clientData, XtPointer callData)
{
   XawIconListReturnStruct	*SelectEntry;
   XawIconList			*FileList;
   FileSelect			*fs = (class FileSelect *)clientData;

   SelectEntry = (XawIconListReturnStruct *)callData;

   // Check if a directory is selected. If so, change to this
   // directory and create a new file list

   switch (fs->CheckFileType(SelectEntry->list_index))
   {
     case FF_DIRECTORY:	// Entry is a directory

	fs->ChangeDir(SelectEntry->list_index);
	if ((FileList = fs->BuildFileList()) != NULL)
	{
	  // Create a new file list. Put the current path name into
	  // the text field for the selected file name.

	  fs->SetFileName(fs->CurrentPath);
	  fs->ChangeList(FileList);
	}
	else	// Error: Can not create file list
	{
	  fs->ResetDir();
	  fs->Warning();
	  fs->PopDown();
	}
        break;

     case FF_FILE:	// Normal file

	fs->SetFileName(fs->GetFileName(SelectEntry->list_index));
	break;

     default:		// Illegal entry ?!
	fs->Warning();
        break;
   }
}


// --- Implementation of class methods ----------------------------------

// **********************************************************************
//
// FileSelect(): The constructor of this class
//		 Creates the file select window
//
// Arguments:	 shell - The root shell widget of this application
//
// **********************************************************************

FileSelect::FileSelect(Widget shell) : FileFunc()
{
  static char	*NoCR = "<Key>Return: no-op()";	   // To ignore CR as input
  XtTranslations TransTab;
  Widget	 panel, fileForm, buttonBox, view, fileLabel;

  tar = NULL;		// We don't have a tar object at this time

// In the first step we create a shell widget to popup and popdown this
// dialogue. On the shell widget we create a panel widget as manager for
// all other widgets

  dialogueShell=XtVaCreatePopupShell("fileDialogShell", transientShellWidgetClass,
		shell, XtNallowShellResize, TRUE, NULL);

  panel       = XtVaCreateManagedWidget("fileDialogPanel", panedWidgetClass, 
		dialogueShell, XtNallowResize, TRUE, NULL);

// On our panel we create a viewport for the file list and two form widgets:
// The first to display the file name and the second for the Ok and Cancel
// button

  view	      = XtVaCreateManagedWidget("listView", viewportWidgetClass, panel,
		XtNshowGrip, FALSE, XtNforceBars, TRUE,
		XtNallowVert, TRUE, XtNallowHoriz, TRUE,
		XtNuseBottom, TRUE, XtNuseRight, TRUE,
		NULL);

  fileForm    = XtVaCreateManagedWidget("fileForm", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, NULL);

  buttonBox   = XtVaCreateManagedWidget("fileButtons", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, NULL);

// This is the list widget for the file names

   fileList   = XtVaCreateManagedWidget("selectList", iconListWidgetClass, view,
		XtNforceColumns, TRUE, NULL);
   XtAddCallback(fileList, XtNcallback, SetFileEntry, (XtPointer)this);

// These widgets are needed to display the currently selected file name

  fileLabel   =	XtVaCreateManagedWidget("selectLabel", labelWidgetClass, fileForm,
		XtNborderWidth, 0, XtNjustify, XtJustifyLeft,
		XtNleft, XtChainLeft, XtNright, XtChainLeft,
		NULL);

  selectFile  = XtVaCreateManagedWidget("fileName", asciiTextWidgetClass, fileForm,
		XtNleft, XtChainLeft, XtNright, XtChainRight,
		XtNeditType, XawtextEdit, XtNfromHoriz, fileLabel,
		NULL);
  TransTab = XtParseTranslationTable(NoCR);
  XtOverrideTranslations(selectFile, TransTab);

// Now we create the needed buttons Ok to open a file and Cancel to abort
// this dialogue

  ok          = XtVaCreateManagedWidget("ok", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		NULL);
  XtAddCallback(ok, XtNcallback, Ok, (XtPointer)this);

  cancel      = XtVaCreateManagedWidget("cancel", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		XtNfromHoriz, ok, NULL);
  XtAddCallback(cancel, XtNcallback, Cancel, (XtPointer)this);
}


// **********************************************************************
//
// PopDown():	Popdown this dialogue shell
//
// Arguments:	None
//
// **********************************************************************

inline void FileSelect::PopDown(void)
{
  XtPopdown(dialogueShell);
}

// **********************************************************************
//
// PopUp():	Popup this dialogue shell at the given position
//
// Arguments:	xPos - X position to popup the dialogue
//		yPos - Y position to popup the dialogue
//
// **********************************************************************

void FileSelect::PopUp(int xPos, int yPos)
{
  XawIconList *FileList;

  // Create a file list of the current directory

  if ((FileList = BuildFileList()) != NULL)
  {
    // Set position and current directory path into tho file dialogue

    XtVaSetValues(dialogueShell, XtNx, xPos, XtNy, yPos, NULL);
    XtVaSetValues(selectFile, XtNstring, CurrentPath, NULL);

    // Provide the file dialogue with the file list and pop it up

    ChangeList(FileList);
    XtPopup(dialogueShell, XtGrabExclusive);
  }
  else Warning();
}

// **********************************************************************
//
// SetFileName(): Set a filename in the selectFile widget
//
// Arguments:	  FName - the filename
//
// **********************************************************************

inline void FileSelect::SetFileName(char *FName)
{
  XtVaSetValues(selectFile, XtNstring, FName, NULL);
}

// **********************************************************************
//
// UpdateTarList(): Update the file list in the main window
//
// Arguments:	    list    - The list
//
// **********************************************************************

inline void FileSelect::UpdateTarList(char *list)
{
  XtVaSetValues(tarListWidget, XtNstring, list, NULL);
}

// **********************************************************************
//
// FinishTarList(): Set archive file name into the label widget of the
//		    main window
//
// Arguments:	    None
//
// **********************************************************************

inline void FileSelect::FinishTarList(void)
{
   char *FName;

   XtVaGetValues(selectFile, XtNstring, &FName, NULL);
   XtVaSetValues(archNameWidget, XtNlabel, FName, NULL);
}

// **********************************************************************
//
// OpenArchive():   Open the tar archive and fill the file list
//		    in the main window.
//
// Arguments:	    FName - File name or NULL
//
// **********************************************************************

void FileSelect::OpenArchive(char *FName)
{
  char *FileList, *FileName = FName;

  if (FileName != NULL) XtVaSetValues(selectFile, XtNstring, FileName, NULL);
  else XtVaGetValues(selectFile, XtNstring, &FileName, NULL);
  if (tar != NULL) delete(tar);
  tar = new HandleTar(FileName);

  if (tar->CheckContent(&FileList) > 0)
  {
    // File is a tar archive and the list is still not complete
    // Set the tar list, close the dialogue and start to continue
    // the work with a work procedure

    XtVaSetValues(tarListWidget, XtNstring, FileList, NULL);
    XtVaSetValues(archNameWidget, XtNlabel, "", NULL);
    (void)XtAppAddWorkProc(XtWidgetToApplicationContext(dialogueShell),
			   ReadTarList, (XtPointer)this);
    PopDown();
  }
  else
  {
    if (FileList != NULL)	// list of tar archive is already complete
    {
      PopDown();
      XtVaSetValues(tarListWidget, XtNstring, FileList, NULL);
      XtVaSetValues(archNameWidget, XtNlabel, FileName, NULL);
      tar->DeleteContent();
    }
    else Warning();		// File is no tar archive
  }
}

// **********************************************************************
//
// GetFileFromArchive(): Extract a file from the tar archive
//
// Arguments:	    listEntry - File entry of the tar list
//
// **********************************************************************

char *FileSelect::GetFileFromArchive(char *listEntry)
{
   return(tar->GetFile(listEntry));
}

// **********************************************************************
//
// ChangeList(): Set a new file list into the icon list widget
//
// Arguments:	 FileList - The list
//
// **********************************************************************

inline void FileSelect::ChangeList(XawIconList *FileList)
{
  XawIconListChange(fileList, FileList, 0, 0, ICON_WIDTH, ICON_HEIGHT, IconDepth, True);
}

// **********************************************************************
//
// Warning():	Gives a warning signal
//
// Arguments:	None
//
// **********************************************************************

inline void FileSelect::Warning(void)
{
  XBell(XtDisplay(dialogueShell), 100);
}

// **********************************************************************
//
// PostInit():	Get the list widget of the root window and create all
//		icons for the icon list, used in the file dialogue.
//		FileSelect needs the list widget to put the list of
//		files of a tar archive into it
//
// Arguments:	FList - The list widget of the root window
//
// **********************************************************************

void FileSelect::PostInit(Widget shell, Widget FList, Widget FNameLabel)
{
  tarListWidget = FList;
  archNameWidget= FNameLabel;
  CreateIcons(shell);
}

// **********************************************************************
//
// ~FileSelect(): The destructor of this class
//		  Destroy the tar object if it exists.
//
// Arguments:	  None
//
// **********************************************************************

FileSelect::~FileSelect(void)
{
   if (tar != NULL) delete(tar);
}

